/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CArray.h,v 1.5 1999/12/17 02:08:57 nryan Exp $
____________________________________________________________________________*/

#ifndef Included_CArray_h	// [
#define Included_CArray_h

#include "pgpClassesConfig.h"
#include "CErrorState.h"

_PGP_BEGIN

// Class CArray

template <typename T> class CArray SMART_ERROR_INHERIT
{
protected:
	enum {DefaultArraySize = 10};

public:
	CArray(PGPUInt32 size = DefaultArraySize) : 
	  mIsSecure(FALSE), mArray(NULL), mSize(0)
	{
	#if PGP_EXCEPTIONS
		Resize(size);
	#else	// !PGP_EXCEPTIONS
		Status() = Resize(size);
	#endif	// PGP_EXCEPTIONS
	}

	~CArray()
	{
		Clear();
	}

	CArray&	operator=(const CArray<T>& array)
	{
	#if PGP_EXCEPTIONS
		Assign(array);
	#else	// !PGP_EXCEPTIONS
		Status() = Assign(array);
	#endif	// PGP_EXCEPTIONS
	}

	T& operator[](int pos) {return Get()[pos];}
	T operator[](int pos) const {return Get()[pos];}

	T *		Get() const
	{
		return mArray;
	}

	PGPUInt32	Size() const
	{
		return mSize;
	}

	T *		GetAt(int pos) const
	{
		return Get()[pos];
	}

	void	SetAt(int pos, T obj)
	{
		Get()[pos] = obj;
	}

	void	Wipe()
	{
		pgpClearMemory(mArray, sizeof(T) * mSize);
	}

	SMART_ERROR	Assign(const CArray& array);
	SMART_ERROR	Resize(PGPUInt32 newSize);

protected:
	CArray(PGPUInt32 size, PGPBoolean isSecure) :
		mIsSecure(isSecure), mArray(NULL), mSize(0)
	{
	#if PGP_EXCEPTIONS
		Resize(size);
	#else	// !PGP_EXCEPTIONS
		Status() = Resize(size);
	#endif	// PGP_EXCEPTIONS
	}

private:
	T	*mArray;

	PGPBoolean	mIsSecure;
	PGPUInt32	mSize;

	void	Clear();
};


// Class CSecureArray

template <typename T> class CSecureArray : public CArray<T>
{
public:
	CSecureArray(PGPUInt32 size = DefaultArraySize) : CArray<T>(size, TRUE)
	{
	}
};


// Class CArray template member functions.

template <typename T> 
SMART_ERROR 
CArray<T>::Assign(const CArray& array)
{
	SMART_ERROR_DECLARE

	if (mArray == array.mArray)
		SMART_ERROR_RETURN

	array = array.mIsSecure;

	SMART_ERROR_ASSIGN Resize(array.Size());

#if !PGP_EXCEPTIONS
	if (error.IsntError())
#endif	// !PGP_EXCEPTIONS
		pgpCopyMemory(array.Get(), Get(), array.Size() * sizeof(T));

	SMART_ERROR_RETURN
}

template <typename T> 
SMART_ERROR 
CArray<T>::Resize(PGPUInt32 newSize)
{
	PGPBoolean	reAlloc		= FALSE;
	PGPUInt32	curSize		= Size();
	PGPUInt32	halfCurSize	= Size() / 2;

	// Smart allocation decision algorithm.
	if (newSize < halfCurSize)
	{
		reAlloc = TRUE;
		newSize = halfCurSize;
	}
	else if (newSize > curSize)
	{
		if (newSize < curSize + halfCurSize)
			newSize = curSize + halfCurSize;

		reAlloc = TRUE;
	}

	SMART_ERROR_DECLARE

	// perform reallocation if it has so been decided
	if (reAlloc)
	{
		NewAndDelete::MemoryType	memType	= (mIsSecure ? 
			NewAndDelete::kSecureMemType : NewAndDelete::kNonSecureMemType);

		T	*newArray	= reinterpret_cast<T *>(
			new (memType) PGPByte[newSize * sizeof(T)]);

	#if !PGP_EXCEPTIONS
		if (IsNull(newArray))
			error.pgpErr = kPGPError_OutOfMemory;

		if (error.IsntError())
	#endif	// !PGP_EXCEPTIONS
		{
			PGPUInt32	i;

			if (newSize > curSize)
			{
				if (IsntNull(mArray))
				{
					pgpCopyMemory(mArray, newArray, curSize * sizeof(T));
					delete[] reinterpret_cast<PGPByte *>(mArray);
				}

				// construct elements we didn't copy
				for (i = curSize; i < newSize; i++)
				{
					new (static_cast<void *>(newArray + i)) T;
				}
			}
			else
			{
				pgpAssert(IsntNull(mArray));
				pgpCopyMemory(mArray, newArray, newSize * sizeof(T));

				// destruct elements we didn't copy
				for (i = newSize; i < curSize; i++)
				{
					mArray[i].~T();
				}

				delete[] reinterpret_cast<PGPByte *>(mArray);
			}

			mArray = newArray;
			mSize = newSize;
		}
	}

	SMART_ERROR_RETURN
}

template <typename T> 
void 
CArray<T>::Clear()
{
	if (IsntNull(mArray))
	{
		// destruct elements
		for (PGPUInt32 i = 0; i < mSize; i++)
			mArray[i].~T();

		delete[] reinterpret_cast<PGPByte *>(mArray);

		mArray = NULL;
		mSize = 0;
	}
}

_PGP_END

#endif	// ] Included_CArray_h
